## HighLevel test Functions TRF Laser
## Command Set according to Flare NX User Command List Firmware 3.0.2, 05.01.2020 
## https://confluence-ext.perkinelmer.com/display/HamRD/Coherent+Flare+NX+343?preview=/339609531/390136829/FlareNX_User_CommandSet_FW3.pdf

import asyncio
import socket
import os
import fnmatch
import json
import time
import configparser
import traceback

from py_pli.pylib import VUnits
from py_pli.pylib import send_msg
from py_pli.pylib import GlobalVar
from urpc_enum.serialparameter import SerialParameter
from fleming.common.firmware_util import get_serial_endpoint


FREQ_MIN = 0.05     # kHz
FREQ_MAX = 3.0      # khz
RESET = '*RST'
SINGLE = '3'        # Single Pulse Mode: Each Trigger: One Pulse
CONT_PULSE = '12'   # Pulse-Train: Trigger starts continous pulse output
GATED = '2'         # Triggered Pulse Train
BURST = '5'         # Fixed Number of Pulses
INFO = '*IDN?'      # Returns Info-String from laser
ENABLE = 'LD:SET:ENA ON'
DISABLE = 'LD:SET:ENA OFF'

# High Level Functions

async def trf_timing_test(frequency_kHz = 0.1):
    # FLE-2105
    mode = SINGLE
    freq = float(frequency_kHz) if frequency_kHz else 0.1
    await trf_init(mode, freq)
    msg = f'Initialize TRF Laser: Single Shot, Frequency = {freq}kHz'
    await send_msg(json.dumps({"result": msg}))
    # start_timer
    # trigger laser
    # wait on sync
    # log time

async def trf_adjust(freq = 0.1):
    # Switch on TRF with low power
    mode = P_TRAIN
    await trf_init(mode, freq)
    await trf_enable()
    msg = f'TRF Laser switched on at low power for adjustment'
    print (msg)
    await send_msg(json.dumps({"result": msg}))
    

# Low Level Functions

async def trf_query(command: str):
    serial = get_serial_endpoint('trf')
    command = command + '\r\n'
    await serial.SetParameter(SerialParameter.ReadTimeout, 1000, timeout=1)
    await serial.ClearReadBuffer(timeout=1)
    await serial.Write(len(command), command.encode('utf-8'), timeout=2)
    rxdata = ''
    try:
        response = await serial.ReadUntil(ord('\r'), timeout=1.1)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except Exception as ex:
        PyLogger.logger.error(ex)

    return bytes(rxdata).decode('utf-8')


async def trf_send(command: str):
    serial = get_serial_endpoint('trf')
    command = command + '\r\n'
    await serial.Write(len(command), command.encode('utf-8'), timeout=2)


async def trf_init(pulse_mode = '3', frequency_Hz = 2.0, num_pulses = 1):
    # Initialize Laser and receive acknowledge string

    # pulse_mode
        # Single = 3
        # Gated = 2
        # Burst = 5
        # Pulse train = 12

    # [ ] tested 
    freq = float(frequency_Hz) if frequency_Hz else 2.0
    mode = str(pulse_mode) if pulse_mode else 3.0
    num = int(num_pulses) if num_pulses else 1

    msg = "TRF Init"
    await send_msg(json.dumps({"result": msg}))
    rxdata = await trf_query('*IDN?')
    msg = f"*IDN?: {rxdata}"
    await send_msg(json.dumps({"result": msg}))
    cmd_mode = 'LD:PULS:MOD ' + mode
    cmd_freq = f'LD:PULS:FRE {freq:3.1f}' 
    cmd_num = f'LD:PULS:NUM {num}'
    await trf_send(cmd_mode)     # single shot
    msg = f'TRF Mode set to {mode}'
    await send_msg(json.dumps({"result": msg}))
    await trf_send(cmd_freq)   # frequency set freq kHz
    msg = f'TRF Frequency set to {freq:3.1f} kHz'
    await send_msg(json.dumps({"result": msg}))
    await trf_send(cmd_num)   # number of pulses set
    msg = f'TRF Number of Pulses set to {num_of_pulses:4.0f}'
    await send_msg(json.dumps({"result": msg}))

async def trf_trigger(duration_s = 0.001):
    # Trigger laser for duration_us microsec
    duration = float(duration_s) if duration_s else 0.001 # 1 millisec
    PyLogger.logger.info("TRF Trigger on !")
    await trf_send('LD:SET:ENA ON')     # enable on
    start_on = time.perf_counter() 
    GlobalVar.set_stop_gc(False)
    while (time.perf_counter() - start_on) < duration:
        if GlobalVar.get_stop_gc():
            await trf_send('LD:SET:ENA OFF')
            msg = "Stopped by user: TRF Laser off"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))
            return msg
    await trf_send('LD:SET:ENA OFF')
    return

async def trf_enable():
    PyLogger.logger.info("TRF Enable !")
    await trf_send('LD:SET:ENA ON')     # enable on
    return

async def trf_disable():
    PyLogger.logger.info("TRF Disable !")
    await trf_send('LD:SET:ENA OFF')    # enable off
    return

async def trf_reset():
    PyLogger.logger.info("TRF Reset")
    rxdata = await trf_query('*RST')    # reset
    PyLogger.logger.info(f"*RST: {rxdata}")
    return
